/*
 * i2c_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * vI2CTask is called by main() to configure the I2C2 module for Master mode.
 * It also creates the read sensor task which handles reading the TMP117 sensor
 * and outputting the results.
 *
 * prvReadSensor starts by powering the TMP117 on the BP-BASSENSORSMKII
 * BoosterPack by turning the GPIO connected to the T_V+ pin high.  This powers
 * the TMP117 sensor.  Afterwards, it takes a reading every 1 millisecond and
 * receives the raw temperature data from the sensor.  Then three different
 * conversion functions are demonstrated to translate and output the result
 * over UART as both an integer and a decimal down to a millidegree C.
 *
 * This example uses UARTprintf for output of UART messages.  UARTprintf is not
 * a thread-safe API and is only being used for simplicity of the demonstration
 * and in a controlled manner.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/i2c.h"
#include "driverlib/pin_map.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
#include "config.h"
#include "TMP117.h"
/*-----------------------------------------------------------*/

/*
 * The notification used by the task.
 */
TaskHandle_t xTaskI2CHandle = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvReadSensor( void *pvParameters );

/*
 * Called by main() to create the I2C application.
 */
void vI2CTask( void );

/*
 * Hardware configuration for the I2C peripheral to read the TMP117 temperature
 * sensor.
 */
static void prvConfigureI2C( void );
/*-----------------------------------------------------------*/

void vI2CTask( void )
{
    /* Configure the I2C1 peripheral to read the TMP117. */
    prvConfigureI2C();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for LED processing task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvReadSensor,
                 "TMP117",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 &xTaskI2CHandle );
}
/*-----------------------------------------------------------*/

static void prvReadSensor( void *pvParameters )
{
int32_t tmp1170Temp;
int32_t tmp1170TempAsIntCelsius;
int32_t tmp1170TempAsMilliCelsius;
float tmp1170TempAsFloatCelsius;
int32_t tmp1170TempAsInteger;
int32_t tmp1170TempAsDecimal;

    /* The TMP117 on the BP-BASSENSORSMKII BoosterPack requires power to
     * operate.  The T_V+ which is Pin 28 on the BoosterPack is the power pin
     * for the TMP117.  This corresponds to PE2 on the EK-TM4C123GXL LaunchPad.
     * This step is not necessary if using the TMP117 in a stand alone
     * application. */
    GPIOPinTypeGPIOOutput(GPIO_PORTE_BASE, GPIO_PIN_2);
    GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_2, GPIO_PIN_2);

    for( ;; )
    {
        /* Delay for 1 second before reading the sensor again. */
        vTaskDelay( pdMS_TO_TICKS( 1000 ) );

        /* Read the TMP117 to get the raw temperature data which needs to be
         * converted. */
        tmp1170Temp = TMP117_read(TMP117_0);

        /* Convert the raw temperature result to different formats.  UARTprintf
         * cannot print float values, so this also demonstrates a possible
         * method to break up the integer and decimal data to make a printable
         * message that can be output correctly by UARTprintf. */
        tmp1170TempAsIntCelsius = TMP117_toIntCelsius(tmp1170Temp);
        tmp1170TempAsMilliCelsius = TMP117_toMilliCelsius(tmp1170Temp);
        tmp1170TempAsInteger = tmp1170TempAsMilliCelsius / 1000;
        tmp1170TempAsFloatCelsius = TMP117_toFloatCelsius(tmp1170Temp);
        tmp1170TempAsDecimal = ((tmp1170TempAsFloatCelsius -
                                                tmp1170TempAsInteger) * 1000);

        /* Display the data that the I2C1 master is transferring.  Observe how
         * there is a difference between the integer display and decimal
         * display when a temperature is above 0.500 degrees. */
        UARTprintf("Temperature in Celsius as integer: %d\n",
                   tmp1170TempAsIntCelsius);
        UARTprintf("Temperature in Celsius with decimals: %d.%03d\n",
                   tmp1170TempAsInteger, tmp1170TempAsDecimal);
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureI2C( void )
{
    /* The I2C1 peripheral must be enabled before use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_I2C1);

    /* Configure the pin muxing for I2C1 functions on port A6 and A7. */
    GPIOPinConfigure(GPIO_PA6_I2C1SCL);
    GPIOPinConfigure(GPIO_PA7_I2C1SDA);

    /* Select the I2C function for these pins.  This function will also
     * configure the GPIO pins pins for I2C operation, setting them to
     * open-drain operation with weak pull-ups.  Consult the data sheet
     * to see which functions are allocated per pin. */
    GPIOPinTypeI2CSCL(GPIO_PORTA_BASE, GPIO_PIN_6);
    GPIOPinTypeI2C(GPIO_PORTA_BASE, GPIO_PIN_7);

    /* Enable and initialize the I2C1 master module.  Use the system clock for
     * the I2C1 module.  The last parameter sets the I2C data transfer rate.
     * If false the data rate is set to 100kbps and if true the data rate will
     * be set to 400kbps.  For this example we will use a data rate of 100kbps.
     */
    I2CMasterInitExpClk(I2C1_BASE, configCPU_CLOCK_HZ, false);
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}

